% PatternSolver.m - Finds the best filter array pattern
close all;     clear all;     clc;
warning off;

%-------------------------------------------------------------------------------
%% Set up a scene for MTF measurement
%-------------------------------------------------------------------------------
wave = (380:10:710)';
scene_mtf = sceneCreate('slantedBar',[],[],wave);

scene_lum  = 75;       % mean luminance (candelas)
scene_fov  = 2.64;     % field of view (degrees)
scene_dist = 10;       % distance of imager from scene (m)

scene_mtf = sceneAdjustLuminance(scene_mtf,scene_lum);
scene_mtf = sceneSet(scene_mtf,'fov',scene_fov);
scene_mtf = sceneSet(scene_mtf,'distance',scene_dist);

vcReplaceAndSelectObject(scene_mtf);

% Show scene window
%sceneWindow;


%-------------------------------------------------------------------------------
%% Set up a scene for Color measurement
%-------------------------------------------------------------------------------
spectrum.wave = wave;
patchSize = 8;
scene_color = sceneCreate('macbethEE_IR',patchSize,spectrum);

scene_color = sceneAdjustLuminance(scene_color,scene_lum);
scene_color = sceneSet(scene_color,'fov',scene_fov);
scene_color = sceneSet(scene_color,'distance',scene_dist);

vcReplaceAndSelectObject(scene_color);

% Show scene window
%sceneWindow;


%-------------------------------------------------------------------------------
%% Optics - MTF
%-------------------------------------------------------------------------------

opt_fnumber = 4;        % f#
opt_flength = 20e-3; % 10e-3; % focal length in metres

% Initialize optics and create irradiance image of specified scene
oi_mtf     = oiCreate;
optics_mtf = oiGet(oi_mtf,'optics');
optics_mtf = opticsSet(optics_mtf,'fnumber',opt_fnumber);
optics_mtf = opticsSet(optics_mtf,'focallength',opt_flength);
optics_mtf = opticsSet(optics_mtf,'offaxis','skip');
oi_mtf     = oiSet(oi_mtf,'optics',optics_mtf);
oi_mtf     = oiCompute(scene_mtf,oi_mtf);

vcReplaceAndSelectObject(oi_mtf);
% Uncomment the following line to see optics window
% oiWindow;


%-------------------------------------------------------------------------------
%% Optics - Color
%-------------------------------------------------------------------------------

% Initialize optics and create irradiance image of specified scene
oi_color     = oiCreate;
optics_color = oiGet(oi_color,'optics');
optics_color = opticsSet(optics_color,'fnumber',opt_fnumber);
optics_color = opticsSet(optics_color,'focallength',opt_flength);
optics_color = opticsSet(optics_color,'offaxis','skip');
oi_color     = oiSet(oi_color,'optics',optics_color);
oi_color     = oiCompute(scene_color,oi_color);

vcReplaceAndSelectObject(oi_color);
% Uncomment the following line to see optics window
% oiWindow;


%-------------------------------------------------------------------------------
%% Create sensor
%-------------------------------------------------------------------------------

% Set the parameters for a multiband color sensor
nSensors = 7;
minW = 380;
maxW = 710;
cPos = linspace(minW,maxW,nSensors+2);
cPos = cPos(2:end-1);
width = ones(size(cPos))*(cPos(2)-cPos(1))/2;


%-------------------------------------------------------------------------------
%% Build a default sensor
%-------------------------------------------------------------------------------
% Add the multiple band filters
cfType = 'gaussian'; 
filterSpectra = sensorColorFilter(cfType, wave, cPos, width);

% Use some subset of these names
allNames = {'b1','g1','r1','x1','i1','z1','i2'};
filterNames = cell(1,nSensors);
for ii=1:nSensors, filterNames{ii} = allNames{ii}; end

% figure(1); plot(wave,filterSpectra)

%-------------------------------------------------------------------------------
%% Backup the optics and scenes
%-------------------------------------------------------------------------------
numPatterns = 5000;
save('PatternSolverVariables.mat', 'oi_color', 'oi_mtf', ...
     'wave', 'filterNames', 'filterSpectra', 'numPatterns', 'scene_color', ...
     'scene_mtf');

savedData = struct([]);
for patternIndex = 1:numPatterns,
   disp(['Pattern index: ' num2str(patternIndex)])
   
   vcReplaceAndSelectObject(scene_mtf);
   vcReplaceAndSelectObject(oi_mtf);
   
   %----------------------------------------------------------------------------
   %% Define the Sensor
   %----------------------------------------------------------------------------
   sensor = sensorCreate;
   wave = sceneGet(scene_mtf,'wave');
   sensor = sensorSet(sensor,'wave',wave);
   sensor = sensorSet(sensor,'filterSpectra',filterSpectra);
   sensor = sensorSet(sensor,'filterNames',filterNames);
   
   %----------------------------------------------------------------------------
   %% Set the spatial pattern
   %----------------------------------------------------------------------------
   p = [];
   while length(unique(p(:))) ~= nSensors
       p = round(rand(3,3)*(nSensors-1) + 1);
   end
   sensor = sensorSet(sensor,'pattern',p);
   %sensorShowCFA(sensor);

   
   %----------------------------------------------------------------------------
   %% MTF-SPECIFIC SENSOR COMPUTATIONS
   %----------------------------------------------------------------------------   
   % Match the sensor size to the scene FOV
   sensor = sensorSetSizeToFOV(sensor,sceneGet(scene_mtf,'fov'));
   
   % Compute the image and bring it up.
   sensor = sensorCompute(sensor,oi_mtf);
   vcAddAndSelectObject(sensor);
   %sensorImageWindow;


   %----------------------------------------------------------------------------
   %% Create and set the processor window
   %----------------------------------------------------------------------------
   vci = vcImageCreate;
   vci = imageSet(vci,'scaledisplay',1);
   vci = imageSet(vci,'renderGamma',0.6);

   % Use the linear transformation derived from sensor space (above) to
   % display the RGB image in the processor window.
   vci = imageSet(vci,'colorConversionMethod','MCC Optimized');
   vci = imageSet(vci,'colorBalanceMethod','Gray World');% 
   %vci = imageSet(vci,'internalCS','XYZ');
   
   vci = vcimageCompute(vci,sensor);
   vcAddAndSelectObject(vci);
   %vcimageWindow
   
   
   %----------------------------------------------------------------------------
   %% Calculate an MTF
   %----------------------------------------------------------------------------
   masterRect = [ 106    43   124   192]; 
   roiLocs = vcRect2Locs(masterRect);
   barImage = vcGetROIData(vci,roiLocs,'results');
   c = masterRect(3)+1;
   r = masterRect(4)+1;
   barImage = reshape(barImage,r,c,3);

   pixel = sensorGet(sensor,'pixel');
   dx = pixelGet(pixel,'width','mm');

   % Run the ISO 12233 code.  The results are stored in the window.
   [results, fitme, esf] = ieISO12233(barImage,dx);
   
   
   %----------------------------------------------------------------------------
   %% Clear some variables
   %----------------------------------------------------------------------------
   clear vci;
   clear barImage;
   clear sensor;
   clear oi_mtf;
   
   
   
   %----------------------------------------------------------------------------
   %% Define the Sensor
   %----------------------------------------------------------------------------
   vcReplaceAndSelectObject(scene_color);
   vcReplaceAndSelectObject(oi_color);
   
   sensor = sensorCreate;
   wave = sceneGet(scene_color,'wave');
   sensor = sensorSet(sensor,'wave',wave);
   sensor = sensorSet(sensor,'filterSpectra',filterSpectra);
   sensor = sensorSet(sensor,'filterNames',filterNames);
   sensor = sensorSet(sensor,'pattern',p);
   
   %----------------------------------------------------------------------------
   %% COLOR-SPECIFIC SENSOR COMPUTATIONS
   %----------------------------------------------------------------------------
   % Match the sensor size to the scene FOV
   sensor = sensorSetSizeToFOV(sensor,sceneGet(scene_color,'fov'));

   % Compute the image and bring it up.
   sensor = sensorCompute(sensor,oi_color);
   vcAddAndSelectObject(sensor);
   %sensorImageWindow;
   
   
   
   sensorLocs = [...
   245     1
   247   329
    29   329
    27     2];
   if ieNotDefined('sensorLocs')
      [L,sensorLocs] = sensorCCM(vcGetObject('sensor'),'macbeth');
   else
      % If you already figured out sensorLocs, you can just run this
      [L,sensorLocs] = sensorCCM(vcGetObject('sensor'),'macbeth',sensorLocs);
   end
   
   %----------------------------------------------------------------------------
   %% Create and set the processor window
   %----------------------------------------------------------------------------
   vci = vcImageCreate;
   vci = imageSet(vci,'scaledisplay',1);
   vci = imageSet(vci,'renderGamma',0.6);

   % Use the linear transformation derived from sensor space (above) to
   % display the RGB image in the processor window.
   vci = imageSet(vci,'colorConversionTransform',L);
   vci = imageSet(vci,'colorBalanceTransform',[]);% 
   vci = imageSet(vci,'ics2displayTransform',[]);  % ICS 2 display
   vci = imageSet(vci,'colorConversionMethod','Current matrix');
   vci = imageSet(vci,'internalCS','Sensor');

   
   vci = vcimageCompute(vci,sensor);
   vcAddAndSelectObject(vci);
   %vcimageWindow


   %----------------------------------------------------------------------------
   %% Create and set the processor window
   %----------------------------------------------------------------------------
   pointLoc = ...
    [248    2
    246   328
    24    327
    24     3];
   [macbethLAB, macbethXYZ, deltaE] = macbethColorError(vci,'D65',pointLoc);
   [yNoise,mRGB] = macbethLuminanceNoise(vci,pointLoc);
   deltaE_avg = mean(deltaE);
   yNoise_avg = mean(yNoise);
   
   %----------------------------------------------------------------------------
   %% Save the results
   %----------------------------------------------------------------------------
   A.pattern = uint8(p);
   A.mtf50 = results.mtf50;
   A.avgDeltaE = deltaE_avg;
   A.noise = mean(yNoise_avg);
   
   savedData = [savedData; A];
   save('./PatternSolverData.mat', 'savedData', 'cPos', 'nSensors');
   
   %----------------------------------------------------------------------------
   %% Clear everything and start fresh
   %----------------------------------------------------------------------------  
   save('PatternSolverIndex.mat', 'patternIndex');
   close all;  clear all;  clc;
   load PatternSolverVariables.mat;
   load PatternSolverData.mat;
   load PatternSolverIndex.mat;
end

PatternProcessor;

% End of File
